//Environment Lib Version v2.7

//This file is part of Basic Shader.
//Read LICENSE First at composite.fsh

const float pi = 3.1415926535897932834919;

vec3 ACESTonemap(vec3 color) {

	const float A = 2.51f;
	const float B = 0.03f;
	const float C = 2.43f;
	const float D = 0.59f;
	const float E = 0.14f;
	
	color = (color * (A * color + B)) / (color * (C * color + D) + E);
	
    return color;
	
}

vec3 doTone(vec3 color, vec3 sunPos, float W) {

    float sunfade = 1.0 - clamp(1.0 - exp(-(sunPos.z / 500.0)), 0.0, 1.0);
	
    color = ACESTonemap(pow(color * 7.241657387 / W, vec3(2.2f)));
    color = pow(color, vec3(1.0 / (1.2 + 1.2 * sunfade)));
	
    return color;
	
}

vec3 doNightSky(vec3 color) {
	
	float amount = 0.05f + 0.2f * timeSkyDark;
	float colorDesat = dot(color, vec3(1.0f)) + 0.10 * timeMidnight;
	
	return mix(color, vec3(colorDesat) * vec3(0.2f, 0.4f, 1.0f), timeMidnight * amount);
	
}

//Atmospheric.
float SunIntensity(float zenithAngleCos, float sunIntensity, float cutoffAngle, float steepness) {
	return sunIntensity * max(0.0, 1.0 - exp(-((cutoffAngle - acos(zenithAngleCos)) / steepness)));
}

float RayleighPhase(float cosViewSunAngle) {
	return (3.0 / (16.0 * pi)) * (1.0 + pow(max(cosViewSunAngle, 0.0), 2.0));
}

float MiePhase(float cosViewSunAngle, float g) {
	return (1.0 / (4.0 * pi)) * ((1.0 - pow(g, 2.0)) / pow(1.0 + pow(g, 2.0) - 2.0 * g * cosViewSunAngle, 1.5));
}

vec3 totalRayleigh(vec3 lambda, float n, float N, float pn){
	return (24.0 * pow(pi, 3.0) * pow(pow(n, 2.0) - 1.0, 2.0) * (6.0 + 3.0 * pn))
	/ (N * pow(lambda, vec3(4.0)) * pow(pow(n, 2.0) + 2.0, 2.0) * (6.0 - 7.0 * pn));
}

vec3 totalMie(vec3 lambda, vec3 K, float T, float v) {
	float c = (0.2 * T) * 10E-18;
	return 0.434 * c * pi * pow((2.0 * pi) / lambda, vec3(v - 2.0)) * K;
}

float getSun(vec3 fragpos, float sunStrength) {

	float weatherRatio = 1.0 - ambient_noise();
	
	float position	 = dot(normalize(fragpos.xyz + vec3(0.0, 15.0, 0.0)), upPosition);
	float horizonPos = mix(pow(clamp(1.0 - pow(abs(position) * 0.05, 1.5), 0.0, 1.0), 5.0), 1.0, 1.0 - clamp(position + length(position), 0.0, 1.0));

	float sunVector  = max(dot(normalize(fragpos), normalize(sunPosition)), 0.0);
	float sun = smoothstep(0.99, 1.0, pow(sunVector, 5.0)) * (sunStrength - 72 * (timeSunrise + timeSunset));
	      sun = mix(sun, 0.0, horizonPos) * weatherRatio * (1.0f - rainStrength);
	
	return sun;

}

float getSunGlow(vec3 fragpos) {

	float sunglowFactor     = pow(1.0f - dot(normalize(-sunVec + normalize(fragpos)), normalize(fragpos)), 4.0f);
	float antiSunglowFactor = pow(1.0f - dot(normalize( sunVec + normalize(fragpos)), normalize(fragpos)), 5.0f);

	float glow  = 1.0f + pow(sunglowFactor, 2.0f) * (5.0f + timeNoon * 1.0f) * (1.0f - rainStrength);
	      glow *= 1.0f + antiSunglowFactor * 2.0f * (1.0f - rainStrength);
		  
	return glow;

}

vec3 getAtmospheric(vec3 fragpos, vec3 fogColor, float hasSun){

	//Wavelength of the primary colors RGB in nanometers.
	const vec3 primaryWavelengths = vec3(700, 546.1, 435.8) * 1.0E-9;
	
	//Physics constants
	float n  = 1.000278;    //Refractive index of air
	float pn = 0.03;	    //Depolarization factor for standard air
	
	float Na = 6.022141E23; //Avogadro constant
	float R  = 8.314472;    //Gas constant
	float T  = 273.15;      //Thermodynamic temperature of 0 celsius
	
	//Environment variables
	float Tm = 15;          //Suppose that the average temperature is 15 celsius
	float AP = 101325;      //The standard air pressure
	
	//Number of molecules per unit volume for air
	float N  = AP * Na / (Tm + T) / R;

	//Optical length at zenith for molecules
	float rayleighZenithLength 		= 8.50E3;
	float mieZenithLength      		= 2.25E3;
	
	//Properties of scattering
	const float mie                 = 0.80;
	const float turbidity 			= 1.50;
	const float sunIntensity 		= 1000;
	
	const float rayleighCoefficient = 1.5;
	const float mieCoefficient		= 0.005;

	//Earth shadow hack
	float steepness   = 1.5;
	float brightness  = 0.35;
	float curvefactor = 6.0;
	float cutoffAngle = pi * 0.5128205128205128;

	//Cos Angles
	float cosViewSunAngle = dot(normalize(fragpos.xyz), sunVec);
	float cosSunUpAngle   = dot(sunVec, upVec) * 0.9 + 0.1;
	float cosUpViewAngle  = dot(upVec, normalize(fragpos.xyz));

	float sunG = getSunGlow(fragpos.xyz);
	float sunE = SunIntensity(cosSunUpAngle, sunIntensity, cutoffAngle, steepness);

	float zenithAngle = max(0.0, cosUpViewAngle);
	float rayleighOpticalLength = rayleighZenithLength / zenithAngle;
	float mieOpticalLength 		= mieZenithLength 	   / zenithAngle;

	//Calculate scattering
	vec3 rayleighAtX = totalRayleigh(primaryWavelengths, n, N, pn) 			  * rayleighCoefficient;
	vec3 mieAtX 	 = totalMie(primaryWavelengths, sunColor, turbidity, 4.0) * mieCoefficient;

	vec3 rayleighXtoEye = rayleighAtX * RayleighPhase(cosViewSunAngle);
	vec3 mieXtoEye 		= mieAtX 	  * MiePhase(cosViewSunAngle, mie);

	vec3 Fex = exp(-(rayleighAtX * rayleighOpticalLength + mieAtX * mieOpticalLength));
	
	float NightLightScattering = pow(max(1.0 - max(cosUpViewAngle, 0.0), 0.0), 2.0);
		
	//Add up all scattering
	vec3 scattering  = sunE * (1.0 - Fex) * (rayleighXtoEye + mieXtoEye) / (rayleighAtX + mieAtX);
	     scattering *= mix(vec3(1.0), pow(scattering * Fex, vec3(0.5)), clamp(pow(1.0 - cosSunUpAngle, 5.0), 0.0, 1.0));

	vec3 sky  = mix(doTone(scattering, sunVec, pow(sunE / sunIntensity, 1.0 / curvefactor) * sunIntensity * brightness), fogColor, rainStrength);
		 sky += pow(sunColor, vec3(2.2f)) * getSun(fragpos.xyz, sunIntensity * 5.0f) * hasSun;
		 sky += pow(fogColor * 0.5, vec3(2.2f)) * ((NightLightScattering + 0.5 * (1.0 - NightLightScattering)) * clamp(pow(1.0 - cosSunUpAngle, 35.0), 0.0, 1.0));
		 sky  = doNightSky(max(sky, vec3(0.0f)));

	return sky;

}

float getCloudNoise(vec2 uv) {

	float x = mix(cos(500.0 * uv.x), cos(uv.y * 1000.0), 0.5);
	float y = mix(sin(500.0 * uv.y), sin(uv.x * 1000.0), 0.5);
    
    vec2 coord = vec2(x, y);
		 coord = coord.xx * normalize(coord.xy) + vec2(-1.0, 1.0) * coord.yy * normalize(coord.yx);

    return length(uv - vec2(sin(coord.x), cos(coord.y)) - coord) - 0.5;
	
}

const vec2 rotator = vec2(0.91, 1.5);
	
float FbmNoise(vec2 uv) {

    float noise = 0.0;
    
    for (int i = 0; i < 12; i++) {
        uv.xy  = uv.xx * rotator.xy + vec2(-1.0, 1.0) * uv.yy * rotator.yx; 
        noise += 0.5 * getCloudNoise(uv) * pow(0.5, float(i + 1));
    }
	
    return noise;
	
}

vec4 getClouds(vec3 fragpos, vec3 fogColor) {

	#ifndef NO_CLOUDS
	
	//Environment variable.
	float cloudSpeed   = 0.02;
	float cloudCover   = 0.52;
	float cloudOpacity = 0.95;
	
	float sunStrength  = 4.5;
	float moonStrength = 1.2;
	float shadingStrength = 0.01;
	
	float weatherRatio = ambient_noise();
	
	//Get position.
	vec4 worldPos  = gbufferModelViewInverse * vec4(fragpos, 1.0);
	vec4 worldPos2 = gbufferModelViewInverse * vec4(fragpos + lightVector * 20.0, 1.0);

	float position = dot(normalize(fragpos.xyz), upPosition);
	float horizonPos = max(1.0 - abs(position) * 0.03, 0.0);

	float horizonBorder = min((1.0 - clamp(position + length(position), 0.0, 1.0)) + horizonPos, 1.0);

	float sunVector  = max(dot(normalize(fragpos), normalize(sunPosition)),  0.0);
	float moonVector = max(dot(normalize(fragpos), normalize(moonPosition)), 0.0);

	float curvedPos = pow(position, 0.5);

	//Calculate light vectors.
	float sun  = pow(sunVector,   5.0);
	float moon = pow(moonVector, 10.0);

	vec2 wind = vec2(frameTimeCounter * 0.0008) * cloudSpeed;

	mat2 rot = mat2(cos(0.5), sin(0.5), -sin(0.5), cos(0.5));

	vec2 coord  = ((worldPos.xz / worldPos.y)   * curvedPos * 0.00025) * rot + wind * 2.0;
	vec2 coord2 = ((worldPos2.xz / worldPos2.y) * curvedPos * 0.00025) * rot + wind * 2.0;

	float noise   = FbmNoise(coord - wind);
		  noise  += FbmNoise(coord * 4.0 - wind)   / 4.0;
	      noise  += FbmNoise(coord * 12.0 - wind)  / 12.0;
		  noise  += FbmNoise(coord * 34.0 - wind)  / 34.0;

	float noise2  = FbmNoise(coord2 - wind);
		  noise2 += FbmNoise(coord2 * 4.0 - wind)  / 4.0;
		  noise2 += FbmNoise(coord2 * 12.0 - wind) / 12.0;

	cloudCover = mix(cloudCover, 0.8, horizonBorder);
	cloudCover = mix(cloudCover, 0.6, sqrt(weatherRatio));
	
	float dist = length(fragpos.xz + cameraPosition.xz * 0.5);
	cloudCover *= max(0.0f, 1.0f - dist / 4000.0f);

	float cloud        = max(noise  - cloudCover * 1.25, 0.0);
	float cloudShading = max(noise2 - cloudCover * 0.8,  0.0);

	//Apply conditions.
	sunStrength  	*= mix(1.0, 0.08, timeMidnight);
	moonStrength 	*= mix(1.0, 0.25, 1.0f - timeMidnight);

	cloudOpacity = mix(cloudOpacity, cloudOpacity * 2.5, cloudCover);

	vec3 cloudClr = vec3(0.015f * moonStrength + 0.25f * sunStrength * (1.2f - timeSunrise - timeSunset) / (1.0f + timeNoon * 0.2f));
		 cloudClr = mix(cloudClr, sunColor , sun * sunStrength * cloud * (1.0f - timeMidnight));
		 cloudClr = mix(cloudClr, moonColor, moon * moonStrength * cloud * timeMidnight);
		 cloudClr = mix(cloudClr, vec3(length(cloudClr) * 0.5), timeNoon);
		 
	cloudClr *= mix(0.75, 1.0, timeFading);
	cloud    *= mix(0.5,  1.0, timeFading);
    
	cloudClr = mix(cloudClr, fogColor * 0.6, cloudShading * shadingStrength * timeFading * cloudOpacity);

	return vec4(pow(max(cloudClr, vec3(0.0f)), vec3(2.2f)), min(cloud * cloudOpacity, 1.0) * (1.0 - horizonBorder));
	
	#else 
	
	return vec4(0.0f);
	
	#endif
	
}

float getStar(vec3 worldpos){

	float starNoise = fract(sin(dot(texcoord.xy * 4.0f + (worldpos.xy + cameraPosition.xy) / 25000000.0 * vec2(frameTimeCounter / 300.0, frameTimeCounter / 300.0), vec2(18.9898f, 28.633f))) * 4378.5453f);
		  starNoise = pow(starNoise, 1024.0f);
	float horizont     = abs(worldpos.y + cameraPosition.y - texcoord.y);
	float horizont_pos = max(pow(max(1.0 - horizont / 750.0, 0.01), 8.0) - 0.1, 0.0);
	float stars = starNoise * timeMidnight * (0.1 - horizont_pos * 0.1) * 0.1f;
	
	return stars;
	
}